//
// This program is free software: you can redistribute it and/or modify
// it under the terms of the GNU Lesser General Public License as published by
// the Free Software Foundation, either version 3 of the License, or
// (at your option) any later version.
// 
// This program is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
// GNU Lesser General Public License for more details.
// 
// You should have received a copy of the GNU Lesser General Public License
// along with this program.  If not, see http://www.gnu.org/licenses/.
//

#ifndef RIPNGROUTING_H_
#define RIPNGROUTING_H_

#include <omnetpp.h>
#include "UDPSocket.h"
#include "IPv6Address.h"
#include "RoutingTable6.h"
#include "RoutingTable6Access.h"
#include "IInterfaceTable.h"
#include "InterfaceTableAccess.h"

#include "RIPngInterface.h"
#include "RIPngRoutingTableEntry.h"
#include "xmlParser.h"

#include "UDPControlInfo_m.h"
#include "RIPngMessage_m.h"
#include "RIPngTimer_m.h"

/**
 *  Represents routing protocol RIPng. RIPng uses UDP and communicates
 *  using UDPSocket.
 */
class RIPngRouting : public cSimpleModule
{
  protected:
    const char  *deviceId;   ///< Id of the device which contains this routing process.
    std::string hostName;    ///< Device name from the network topology.
    std::string routerText;  ///< Text with hostName for ev printing.

    UDPSocket   socket;
    int         localPort, destPort;
    int         iRouteTimeout;
    int         iRouteGarbageCollectionTimeout;
    IPv6Address RIPngAddress;

    // statistics
    int numSent;
    int numReceived;

    IInterfaceTable*                                ift;                ///< Provides access to the interface table.
    RoutingTable6*                                  rt;                 ///< Provides access to the IPv6 routing table.
    std::map<std::string,RIPng::RoutingTableEntry*> routingTable;       ///< The RIPng routing table - contains more information than the one in the IP layer.
    std::vector<RIPng::Interface*>                  enabledInterfaces;  ///< Interfaces which has allowed RIPng

  protected:
    typedef map<std::string,RIPng::RoutingTableEntry*>::iterator RoutingTableIt;
    unsigned long                   getRoutingTableEntryCount() const  { return routingTable.size(); }
    RIPng::RoutingTableEntry*       getRoutingTableEntry(std::string &addr)  { return routingTable[addr]; }
    void                            addRoutingTableEntry(RIPng::RoutingTableEntry* entry) { routingTable[entry->getDestPrefix().str()] = entry; }
    void                            updateRoutingTableEntry();

    unsigned long            getEnabledInterfacesCount() const  { return enabledInterfaces.size(); }
    RIPng::Interface*        getEnabledInterface(unsigned long i)  { return enabledInterfaces[i]; }
    const RIPng::Interface*  getEnabledInterface(unsigned long i) const  { return enabledInterfaces[i]; }
    void                     addEnabledInterface(RIPng::Interface *interface) { enabledInterfaces.push_back(interface); }

  public:
    void showRoutingTable();

  protected:
    //-- GENERAL METHODS
    RIPngMessage *createMessage();
    void setSocketOptions();

    /**
     * Loads the configuration of the RIPng process from the config XML.
     * @param config [in] The path of the XML config file.
     * @return True if the configuration was succesfully loaded.
     */
    bool loadConfigFromXML(const char *configFileName);

    /**
     * Get the RIPng status of the interface.
     * @param interface [in].
     * @return "enable" if RIPng is enabled on the interface, otherwise "disable".
     */
    const char *getInterfaceRIPngStatus(cXMLElement *interface);

    /**
     * Enables RIPng on the interface.
     * @param interfaceName [in] The interface name on which to enable RIPng.
     */
    void enableRIPngOnInterface(const char *interfaceName);

    /**
     * Adds unicast prefixes (not link-local) obtained from interface to RIPng routing table
     * @param interface [in] RIPng interface, from which to add prefixes
     * @see InterfaceTable
     */
    void addPrefixesFromInterfaceToRT(RIPng::Interface *RIPngInterface);

    //-- OUTPUT PROCESSING
    /**
     * Creates and sends regular update messages-
     * if addr is RIPng multicast address then split horizon processing is used
     * and the message is sent to every ripng connected network with link-local
     * address as a source (port param should be RIPng port)
     * if addr is UNICAST address then ALL routes are included and the message is
     * sent only to that UNICAST address with global unicast address as a source and
     * to the the given port
     * @param addr [in] destination address of the message
     * @param port [in] destination port of the message
     */
    void sendRegularUpdateMessage(IPv6Address &addr, int port);

    /**
     *  Creates and sends triggered update messages -
     *  only routes with change flag set are included in the message.
     *  The message is sent to RIPng mutlicast address.
     */
    void sendTriggeredUpdateMessage();

    /**
     *  This method is used by sendRegularUpdateMessage() and sendTriggeredUpdateMessage()
     *  @param msg [in] message to be send
     *  @param addr [in] destination address
     *  @param port [in] destination port
     *  @param outInterface [in] from which interface should be message sent
     *  @see sendRegularUpdateMessage()
     *  @see sendTriggeredUpdateMessage()
     */
    void sendMessage(RIPngMessage *msg, IPv6Address &addr, int port, int outInterface);

    //-- GENERATING RESPONSE MESSAGES
    void makeResponse(/*adresa na kterou se dela response - unicast, multicast(FF02::9)*/);
    void makeTriggeredUpdate();
    void clearRoutesChangeFlag();

    void rebuildRoutingTable();

    //-- INPUT PROCESSING
    void processMessage(RIPngMessage *msg);

    //-- RESPONSE PROCESSING
    void processResponse();
    // Unnecessary? virtual void checkMessageValidity();
    void processRTEs();
    // Unnecessary? virtual void checkRTE();
    // Unnecessary? virtual void logBadRTE();

    //-- REQUEST PROCESSING
    void processRequest();

    //-- TIMEOUTS
    void routeTimeout();
    void routeGarbageCollectionTimeout();
    void triggeredUpdateTimer();

    void startRouteDeletionProcess();
    void deleteRoute();

  protected:
    virtual int numInitStages() const {return 4;}
    virtual void initialize(int stage);
    virtual void handleMessage(cMessage *msg);
};

#endif /* RIPNGROUTING_H_ */
